// background.js — Lingowl Chat 多语言沟通助手 v1.5.0.0
// 功能：DeepSeek 翻译 + 多轮上下文 + 双语最优回复 + 防重复调用
// v1.5.0.0 更新：通用版本，支持所有企业的跨语言客服需求

// ======= 基本配置 =======
async function getConfig() {
  return new Promise((resolve) => {
    chrome.storage.sync.get(
      {
        apiKey: "",
        targetLang: "zh",
        enable: true,
        whisperModel: "whisper-1",
        chatModel: "deepseek-chat",
        companyName: "",
        companyDescription: "",
        knowledgeBaseUrl: "https://open.lingowl.com/json/",
        replyTemplates: {},
        quickPhrases: {
          greeting: ["Hello! How can I help you today?"],
          product: ["Thank you for your interest in our product. Let me provide you with more details."],
          meeting: ["I'd like to schedule a meeting with you. When would be convenient for you?"]
        },
        updateCheckUrl: "https://open.lingowl.com/lingowl-chat/version.json",
        lastSeenVersion: ""
      },
      resolve
    );
  });
}

// ======= 防重复请求机制 =======
const pendingRequests = new Map(); // 防止同一消息多次请求
function msgFingerprint(text) {
  return text.toLowerCase().trim().replace(/\s+/g, " ").slice(0, 150);
}

// ======= DeepSeek 翻译 =======
async function deepseekTranslateSafe(text, apiKey, targetLang = "zh") {
  const key = msgFingerprint(text) + "::" + targetLang;
  if (pendingRequests.has(key)) {
    console.log("[Helport Copilot] ⏳ 等待已有翻译返回:", key);
    return pendingRequests.get(key);
  }

  const promise = (async () => {
    // 将语言代码转换为语言名称
    const languageNameMap = {
      'zh': '中文',
      'en': '英文',
      'es': '西班牙语',
      'fr': '法语',
      'de': '德语',
      'it': '意大利语',
      'pt': '葡萄牙语',
      'ru': '俄语',
      'ja': '日语',
      'ko': '韩语',
      'ar': '阿拉伯语',
      'hi': '印地语',
      'th': '泰语',
      'vi': '越南语',
      'id': '印尼语',
      'ms': '马来语',
      'tr': '土耳其语',
      'pl': '波兰语',
      'nl': '荷兰语',
      'sv': '瑞典语',
      'no': '挪威语',
      'da': '丹麦语',
      'fi': '芬兰语',
      'el': '希腊语',
      'he': '希伯来语',
      'cs': '捷克语',
      'hu': '匈牙利语',
      'ro': '罗马尼亚语',
      'bg': '保加利亚语',
      'hr': '克罗地亚语',
      'sk': '斯洛伐克语',
      'sl': '斯洛文尼亚语',
      'et': '爱沙尼亚语',
      'lv': '拉脱维亚语',
      'lt': '立陶宛语',
      'uk': '乌克兰语',
      'be': '白俄罗斯语',
      'sr': '塞尔维亚语',
      'mk': '马其顿语',
      'sq': '阿尔巴尼亚语',
      'bs': '波斯尼亚语',
      'me': '黑山语',
      'mo': '摩尔多瓦语'
    };
    
    const languageName = languageNameMap[targetLang] || '英文';
    const prompt = `你是一个专业的翻译工具。请将以下文本翻译成${languageName}。

要求：
1. 只返回翻译结果
2. 不要添加任何解释、前缀、后缀或额外内容
3. 不要使用"回复"、"翻译"等词汇
4. 直接输出翻译后的文本

原文：`;
    
    console.log("[Lingowl Chat] 🔄 翻译请求 - 目标语言:", languageName, "语言代码:", targetLang);
    console.log("[Lingowl Chat] 🔄 翻译请求 - 提示词:", prompt);
    
    const res = await fetch("https://api.deepseek.com/v1/chat/completions", {
      method: "POST",
      headers: {
        Authorization: `Bearer ${apiKey}`,
        "Content-Type": "application/json"
      },
      body: JSON.stringify({
        model: "deepseek-chat",
        messages: [
          { role: "system", content: prompt },
          { role: "user", content: text }
        ],
        temperature: 0.1
      })
    });

    const data = await res.json();
    pendingRequests.delete(key);
    const rawResult = data.choices?.pop()?.message?.content?.trim() || "";
    let result = rawResult;
    
    // 清理可能的多余内容
    if (result) {
      // 移除常见的多余词汇
      result = result.replace(/^(回复|翻译|译文|结果|答案)[：:]\s*/i, '');
      result = result.replace(/^(Reply|Translation|Result|Answer)[:\s]*/i, '');
      result = result.replace(/^[^a-zA-Z\u4e00-\u9fa5]*/, ''); // 移除开头的非字母非中文字符
      result = result.trim();
    }
    
    console.log("[Lingowl Chat] ✅ 翻译完成 - 原始结果:", rawResult);
    console.log("[Lingowl Chat] ✅ 翻译完成 - 清理后结果:", result);
    return result;
  })();

  pendingRequests.set(key, promise);
  return promise;
}

// ======= 版本更新机制 =======
function compareVersions(a, b) {
  const pa = String(a).split('.').map((n) => parseInt(n, 10) || 0);
  const pb = String(b).split('.').map((n) => parseInt(n, 10) || 0);
  for (let i = 0; i < Math.max(pa.length, pb.length); i++) {
    const da = pa[i] || 0;
    const db = pb[i] || 0;
    if (da > db) return 1;
    if (da < db) return -1;
  }
  return 0;
}

async function notifyUpdate(title, message, url) {
  try {
    const id = `lingowl-update-${Date.now()}`;
    chrome.notifications.create(id, {
      type: 'basic',
      iconUrl: 'icons/icon128.png',
      title,
      message,
      priority: 2
    });
    if (url) {
      chrome.notifications.onClicked.addListener((nid) => {
        if (nid === id) {
          chrome.tabs.create({ url });
        }
      });
    }
  } catch (e) {
    console.log('[Lingowl Chat] 通知创建失败:', e?.message || e);
  }
}

async function checkForUpdates(trigger = 'alarm') {
  const manifest = chrome.runtime.getManifest();
  const current = manifest.version;
  const { updateCheckUrl, lastSeenVersion } = await getConfig();
  
  console.log('[Lingowl Chat] 🔍 版本检查开始:', {
    trigger,
    current,
    updateCheckUrl,
    lastSeenVersion
  });
  
  if (!updateCheckUrl) {
    console.log('[Lingowl Chat] ⚠️ 未配置更新检查URL');
    return;
  }
  
  try {
    console.log('[Lingowl Chat] 📡 正在获取远程版本信息...');
    const res = await fetch(updateCheckUrl, { cache: 'no-cache' });
    const data = await res.json().catch(() => ({}));
    const latest = data.version || '';
    const changelogUrl = data.changelog || data.url || 'https://open.lingowl.com/lingowl-chat/';

    console.log('[Lingowl Chat] 📊 版本比较结果:', {
      current,
      latest,
      changelogUrl,
      comparison: compareVersions(latest, current)
    });

    if (latest && compareVersions(latest, current) > 0) {
      console.log('[Lingowl Chat] 🎉 发现新版本，发送通知');
      await notifyUpdate(
        'Lingowl Chat 有新版本可用',
        `当前版本 ${current}，最新版本 ${latest}。点击查看更新说明。`,
        changelogUrl
      );
    } else if (!lastSeenVersion || compareVersions(current, lastSeenVersion) > 0) {
      // 首次安装或扩展自身更新后的一次性提示
      console.log('[Lingowl Chat] 🎉 首次安装或本地更新，发送通知');
      await notifyUpdate(
        'Lingowl Chat 已更新',
        `已升级至版本 ${current}。点击查看更新说明。`,
        changelogUrl
      );
      chrome.storage.sync.set({ lastSeenVersion: current });
    } else {
      console.log('[Lingowl Chat] ✅ 版本检查完成，无需更新');
    }
  } catch (e) {
    console.log('[Lingowl Chat] ❌ 版本检查失败:', e?.message || e);
  }
}

// 安装/更新时触发
chrome.runtime.onInstalled.addListener(async (details) => {
  try {
    const manifest = chrome.runtime.getManifest();
    const current = manifest.version;
    if (details.reason === 'install') {
      await notifyUpdate(
        '欢迎使用 Lingowl Chat',
        `已安装版本 ${current}。点击打开设置开始配置。`,
        chrome.runtime.getURL('popup/popup.html')
      );
      chrome.storage.sync.set({ lastSeenVersion: current });
    } else if (details.reason === 'update') {
      // 安装更新后，标记版本并提示一次
      chrome.storage.sync.get({ lastSeenVersion: '' }, async (cfg) => {
        if (!cfg.lastSeenVersion || compareVersions(current, cfg.lastSeenVersion) > 0) {
          await notifyUpdate(
            'Lingowl Chat 已更新',
            `已升级至版本 ${current}。点击查看更新说明。`,
            'https://open.lingowl.com/lingowl-chat/changelog'
          );
          chrome.storage.sync.set({ lastSeenVersion: current });
        }
      });
    }
  } catch (e) {
    console.log('[Lingowl Chat] onInstalled 处理失败:', e?.message || e);
  }

  // 创建每日检查定时器（若不存在）
  chrome.alarms.get('lingowl_update_check', (alarm) => {
    if (!alarm) {
      chrome.alarms.create('lingowl_update_check', { periodInMinutes: 60 * 24 });
    }
  });
});

// 启动时也进行一次检查
chrome.runtime.onStartup?.addListener(() => {
  checkForUpdates('startup');
});

// 定时器回调
chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name === 'lingowl_update_check') {
    checkForUpdates('alarm');
  }
});

// ======= 检测语言 =======
async function detectLanguage(text, apiKey) {
  const prompt = `Detect language name only (English, Chinese, Spanish, etc.) of this text: "${text}"`;
  const res = await fetch("https://api.deepseek.com/v1/chat/completions", {
    method: "POST",
    headers: {
      Authorization: `Bearer ${apiKey}`,
      "Content-Type": "application/json"
    },
    body: JSON.stringify({
      model: "deepseek-chat",
      messages: [{ role: "user", content: prompt }],
      temperature: 0.1
    })
  });
  const data = await res.json();
  return data.choices?.[0]?.message?.content?.trim()?.toLowerCase() || "english";
}

// ======= 生成双语最优回复 =======
async function generateSmartReplySafe(text, context, apiKey, companyInfo, detectedLanguage) {
  const key = "reply::" + msgFingerprint(text);
  if (pendingRequests.has(key)) {
    console.log("[Lingowl Chat] 🧩 已有最优回复生成中:", key);
    return pendingRequests.get(key);
  }

  const promise = (async () => {
    const { companyName, companyDescription, replyTemplates } = companyInfo;
    const companySummary = companyDescription || JSON.stringify(replyTemplates).slice(0, 1500);

    // 根据检测到的语言调整提示词
    const languageName = detectedLanguage || "English";
    console.log("[Lingowl Chat] 🎯 检测到客户语言:", languageName);

    const systemPrompt = `
You are a bilingual (English and Chinese) professional customer service manager.
${companyName ? `You represent ${companyName} company.` : 'You represent a professional service company.'}
${companySummary ? `Company information: ${companySummary}` : ''}

Your goal:
1. Understand the customer's last message and the recent 5 messages.
2. The customer's message is in ${languageName} language.
3. Provide TWO short replies:
   (a) One in the SAME language as the customer (${languageName}) - this will be the "reply_foreign"
   (b) One in natural, polite Chinese - this will be the "reply_cn"
4. Be concise and natural like a human service manager (not too formal or promotional).
5. If the message is off-topic (not business related), just give a friendly polite reply.
6. IMPORTANT: Write replies from the FIRST PERSON perspective (I received, I understand, etc.) since the user is the one who received the message.
7. Do NOT include explanations or JSON — just output exactly this format:

Foreign reply (in ${languageName}):
...
Chinese reply:
...
`;

    const contextText = context?.map((c) => `${c.role}: ${c.text}`).join("\n") || "";
    const userPrompt = `${contextText}\n\nCustomer message (in ${languageName}): ${text}`;

    const res = await fetch("https://api.deepseek.com/v1/chat/completions", {
      method: "POST",
      headers: {
        Authorization: `Bearer ${apiKey}`,
        "Content-Type": "application/json"
      },
      body: JSON.stringify({
        model: "deepseek-chat",
        messages: [
          { role: "system", content: systemPrompt },
          { role: "user", content: userPrompt }
        ],
        temperature: 0.6
      })
    });

    const data = await res.json();
    const content = data.choices?.pop()?.message?.content || "";

    const matchForeign = content.match(/Foreign reply \(in \w+\):\s*([\s\S]*?)\nChinese reply:/i) || 
                        content.match(/Foreign reply:\s*([\s\S]*?)\nChinese reply:/i);
    const matchCN = content.match(/Chinese reply:\s*([\s\S]*)/i);

    const result = {
      reply_foreign: matchForeign ? matchForeign[1].trim() : "",
      reply_cn: matchCN ? matchCN[1].trim() : "",
      detected_language: languageName
    };

    console.log("[Lingowl Chat] 🎯 生成回复 - 客户语言:", languageName);
    console.log("[Lingowl Chat] 🎯 生成回复 - 外语版本:", result.reply_foreign);
    console.log("[Lingowl Chat] 🎯 生成回复 - 中文版本:", result.reply_cn);

    pendingRequests.delete(key);
    return result;
  })();

  pendingRequests.set(key, promise);
  return promise;
}

// ======= 主监听逻辑 =======
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  (async () => {
    // 处理打开设置页面的请求
    if (msg?.type === "OPEN_SETTINGS") {
      chrome.tabs.create({ url: chrome.runtime.getURL('popup/popup.html') });
      return;
    }
    
    if (msg?.type === "GET_CONFIG") {
      const cfg = await getConfig();
      sendResponse({ ok: true, cfg });
      return;
    }
    
    if (msg?.type === "CHECK_UPDATE") {
      console.log("[Lingowl Chat] 🔄 手动检查更新");
      await checkForUpdates('manual');
      sendResponse({ ok: true });
      return;
    }

    // === 中文翻译成客户语言（用于编辑功能） ===
    if (msg?.type === "TRANSLATE_TO_FOREIGN") {
      const { text, targetLanguage } = msg;
      const { apiKey } = await getConfig();
      if (!apiKey) {
        sendResponse({ ok: false, error: "Missing API key." });
        return;
      }

      try {
        // 根据目标语言确定翻译目标
        let targetLang = "en"; // 默认英文
        if (targetLanguage && targetLanguage.toLowerCase() !== "english") {
          // 如果是非英语，需要将语言名称转换为语言代码
          const languageCodeMap = {
            'chinese': 'zh',
            'spanish': 'es', 
            'french': 'fr',
            'german': 'de',
            'italian': 'it',
            'portuguese': 'pt',
            'russian': 'ru',
            'japanese': 'ja',
            'korean': 'ko',
            'arabic': 'ar',
            'hindi': 'hi',
            'thai': 'th',
            'vietnamese': 'vi',
            'indonesian': 'id',
            'malay': 'ms',
            'turkish': 'tr',
            'polish': 'pl',
            'dutch': 'nl',
            'swedish': 'sv',
            'norwegian': 'no',
            'danish': 'da',
            'finnish': 'fi',
            'greek': 'el',
            'hebrew': 'he',
            'czech': 'cs',
            'hungarian': 'hu',
            'romanian': 'ro',
            'bulgarian': 'bg',
            'croatian': 'hr',
            'slovak': 'sk',
            'slovenian': 'sl',
            'estonian': 'et',
            'latvian': 'lv',
            'lithuanian': 'lt',
            'ukrainian': 'uk',
            'belarusian': 'be',
            'serbian': 'sr',
            'macedonian': 'mk',
            'albanian': 'sq',
            'bosnian': 'bs',
            'montenegrin': 'me',
            'moldovan': 'mo'
          };
          targetLang = languageCodeMap[targetLanguage.toLowerCase()] || "en";
        }
        
        const translated = await deepseekTranslateSafe(text, apiKey, targetLang);
        sendResponse({ ok: true, translated });
      } catch (e) {
        console.error("[Lingowl Chat] ❌ 中文翻译客户语言失败:", e);
        sendResponse({ ok: false, error: e.message });
      }
      return;
    }

    // === 翻译 + 生成双语回复 ===
    if (msg?.type === "TRANSLATE_TEXT") {
      const { id, text, context } = msg;
      const { apiKey, enable, companyName, companyDescription, knowledgeBaseUrl, replyTemplates } = await getConfig();
      if (!enable || !apiKey) {
        sendResponse({ ok: false, error: "Missing API key or disabled." });
        return;
      }

      const msgKey = msgFingerprint(text);
      if (pendingRequests.has(msgKey)) {
        console.log("[Lingowl Chat] 🚫 已存在同消息任务:", msgKey);
        return;
      }

      try {
        // 先快速翻译并回传局部结果
        const translated = await deepseekTranslateSafe(text, apiKey, "zh");
        chrome.tabs.sendMessage(sender.tab.id, {
          type: "TEXT_TRANSLATION_PARTIAL",
          id,
          text,
          translated
        });

        // 然后继续进行语言检测与最优回复生成
        let companyInfo = { companyName, companyDescription, replyTemplates };
        
        // 如果配置了知识库地址，尝试获取知识库内容
        if (knowledgeBaseUrl && knowledgeBaseUrl !== "https://open.lingowl.com/json/") {
          try {
            const knowledgeBase = await fetch(knowledgeBaseUrl).then((r) => r.json());
            companyInfo = { ...companyInfo, ...knowledgeBase };
            console.log("[Lingowl Chat] 📚 已加载知识库:", knowledgeBaseUrl);
          } catch (error) {
            console.log("[Lingowl Chat] ⚠️ 知识库加载失败，使用本地配置:", error.message);
          }
        }
        
        const detectedLang = await detectLanguage(text, apiKey);
        console.log("[Lingowl Chat] 🎯 检测到客户语言:", detectedLang);
        
        const { reply_foreign, reply_cn, detected_language } = await generateSmartReplySafe(text, context, apiKey, companyInfo, detectedLang);

        chrome.tabs.sendMessage(sender.tab.id, {
          type: "TEXT_TRANSLATION_DONE",
          id,
          text,
          translated,
          reply_foreign,
          reply_cn,
          detected_language: detected_language || detectedLang
        });

        sendResponse({ ok: true });
      } catch (e) {
        console.error("[Lingowl Chat] ❌ 错误:", e);
        chrome.tabs.sendMessage(sender.tab.id, {
          type: "TEXT_TRANSLATION_DONE",
          id,
          text,
          error: e.message
        });
        sendResponse({ ok: false, error: e.message });
      }
    }
  })();

  return true;
});
