// popup.js — Lingowl Chat 通用配置版
const $ = (id) => document.getElementById(id);

// ======== 加载设置 ========
document.addEventListener("DOMContentLoaded", () => {
  chrome.storage.sync.get(
    {
      apiKey: "",
      enable: true,
      companyName: "",
      companyDescription: "",
      knowledgeBaseUrl: "https://open.lingowl.com/json/",
      quickPhrases: {
        greeting: ["Hello! How can I help you today?"],
        product: ["Thank you for your interest in our product. Let me provide you with more details."],
        meeting: ["I'd like to schedule a meeting with you. When would be convenient for you?"]
      }
    },
    (cfg) => {
      $("apiKey").value = cfg.apiKey || "";
      $("enable").checked = cfg.enable ?? true;
      $("companyName").value = cfg.companyName || "";
      $("companyDescription").value = cfg.companyDescription || "";
      $("knowledgeBaseUrl").value = cfg.knowledgeBaseUrl || "https://open.lingowl.com/json/";
      
      // 加载快捷短语
      loadQuickPhrases(cfg.quickPhrases || { greeting: [], product: [], meeting: [] });
      
      // 设置自动保存监听器
      setupAutoSave();
    }
  );
});

// ======== 设置自动保存监听器 ========
function setupAutoSave() {
  // 获取所有需要监听的元素
  const inputs = [
    $("apiKey"),
    $("companyName"), 
    $("companyDescription"),
    $("knowledgeBaseUrl")
  ];
  
  // 为每个输入框添加事件监听
  inputs.forEach(input => {
    if (input) {
      input.addEventListener("input", autoSave);
      input.addEventListener("blur", autoSave);
    }
  });
  
  // 为开关添加事件监听
  if ($("enable")) {
    $("enable").addEventListener("change", autoSave);
  }
  
  // 为快捷短语输入框添加事件监听
  document.addEventListener("input", (e) => {
    if (e.target.classList.contains("phrase-input")) {
      autoSave();
    }
  });
}

// ======== 快捷短语功能 ========
function loadQuickPhrases(phrases) {
  const categories = ['greeting', 'product', 'meeting'];
  
  categories.forEach(category => {
    const container = document.getElementById(`${category}-phrases`);
    container.innerHTML = '';
    
    // 加载已保存的短语
    if (phrases[category] && phrases[category].length > 0) {
      phrases[category].forEach((phrase, index) => {
        addPhraseInput(container, phrase, index);
      });
    }
  });
}

function addPhraseInput(container, value = '', index = null) {
  const inputGroup = document.createElement('div');
  inputGroup.className = 'phrase-input-group';
  
  const input = document.createElement('input');
  input.type = 'text';
  input.className = 'phrase-input';
  input.placeholder = '请输入快捷短语';
  input.value = value;
  
  const deleteBtn = document.createElement('button');
  deleteBtn.type = 'button';
  deleteBtn.className = 'delete-phrase-btn';
  deleteBtn.innerHTML = '×';
  deleteBtn.onclick = () => {
    inputGroup.remove();
  };
  
  inputGroup.appendChild(input);
  inputGroup.appendChild(deleteBtn);
  container.appendChild(inputGroup);
}

function getQuickPhrases() {
  const categories = ['greeting', 'product', 'meeting'];
  const phrases = {};
  
  categories.forEach(category => {
    const container = document.getElementById(`${category}-phrases`);
    const inputs = container.querySelectorAll('.phrase-input');
    phrases[category] = Array.from(inputs)
      .map(input => input.value.trim())
      .filter(value => value.length > 0);
  });
  
  return phrases;
}


// ======== 事件监听器 ========
document.addEventListener('click', (e) => {
  if (e.target.classList.contains('add-phrase-btn')) {
    const category = e.target.getAttribute('data-category');
    const container = document.getElementById(`${category}-phrases`);
    addPhraseInput(container);
  }
});

// ======== 自动保存功能 ========
let saveTimeout = null;

function showSaveSuccess() {
  // 创建成功提示
  const tip = document.createElement("div");
  tip.textContent = "✅ 设置已保存";
  tip.style.position = "fixed";
  tip.style.top = "20px";
  tip.style.right = "20px";
  tip.style.background = "#d4edda";
  tip.style.color = "#155724";
  tip.style.padding = "8px 12px";
  tip.style.borderRadius = "4px";
  tip.style.fontSize = "12px";
  tip.style.boxShadow = "0 2px 4px rgba(0,0,0,0.1)";
  tip.style.zIndex = "10000";
  tip.style.border = "1px solid #c3e6cb";
  
  document.body.appendChild(tip);
  
  // 2秒后移除提示
  setTimeout(() => {
    if (tip.parentNode) {
      tip.parentNode.removeChild(tip);
    }
  }, 2000);
}

function autoSave() {
  // 清除之前的定时器
  if (saveTimeout) {
    clearTimeout(saveTimeout);
  }
  
  // 设置新的定时器，1秒后保存
  saveTimeout = setTimeout(() => {
    const apiKey = $("apiKey").value.trim();
    const enable = $("enable").checked;
    const companyName = $("companyName").value.trim();
    const companyDescription = $("companyDescription").value.trim();
    const knowledgeBaseUrl = $("knowledgeBaseUrl").value.trim();
    const quickPhrases = getQuickPhrases();

    chrome.storage.sync.set({ apiKey, enable, companyName, companyDescription, knowledgeBaseUrl, quickPhrases }, () => {
      showSaveSuccess();
    });
  }, 1000);
}


// ======== 手动保存设置 ========
$("save").addEventListener("click", () => {
  const apiKey = $("apiKey").value.trim();
  const enable = $("enable").checked;
  const companyName = $("companyName").value.trim();
  const companyDescription = $("companyDescription").value.trim();
  const knowledgeBaseUrl = $("knowledgeBaseUrl").value.trim();
  const quickPhrases = getQuickPhrases();

  chrome.storage.sync.set({ apiKey, enable, companyName, companyDescription, knowledgeBaseUrl, quickPhrases }, () => {
    showSaveSuccess();
    // 延迟关闭窗口，让用户看到成功提示
    setTimeout(() => {
      window.close();
    }, 1500);
  });
});

// ======== 打开 WhatsApp Web ========
$("openWhatsApp").addEventListener("click", () => {
  chrome.tabs.create({ url: "https://web.whatsapp.com/" });
  window.close();
});

// ======== 打开 LinkedIn ========
$("openLinkedIn").addEventListener("click", () => {
  chrome.tabs.create({ url: "https://www.linkedin.com/messaging/" });
  window.close();
});

// ======== 社交媒体链接 ========
// 抖音
$("openDouyin").addEventListener("click", () => {
  chrome.tabs.create({ url: "https://www.douyin.com/" });
  window.close();
});

// 微博
$("openWeibo").addEventListener("click", () => {
  chrome.tabs.create({ url: "https://weibo.com/" });
  window.close();
});

// 小红书
$("openXiaohongshu").addEventListener("click", () => {
  chrome.tabs.create({ url: "https://www.xiaohongshu.com/" });
  window.close();
});

// B站
$("openBilibili").addEventListener("click", () => {
  chrome.tabs.create({ url: "https://www.bilibili.com/" });
  window.close();
});

// 1688
$("open1688").addEventListener("click", () => {
  chrome.tabs.create({ url: "https://www.1688.com/" });
  window.close();
});

// Facebook
$("openFacebook").addEventListener("click", () => {
  chrome.tabs.create({ url: "https://www.facebook.com/messages/" });
  window.close();
});

// Twitter
$("openTwitter").addEventListener("click", () => {
  chrome.tabs.create({ url: "https://twitter.com/messages" });
  window.close();
});

// TikTok
$("openTikTok").addEventListener("click", () => {
  chrome.tabs.create({ url: "https://www.tiktok.com/messages" });
  window.close();
});

// Alibaba
$("openAlibaba").addEventListener("click", () => {
  chrome.tabs.create({ url: "https://www.alibaba.com/" });
  window.close();
});

// ======== 检查更新 ========
$("checkUpdate").addEventListener("click", async () => {
  const button = $("checkUpdate");
  const originalText = button.textContent;
  
  button.textContent = "检查中...";
  button.disabled = true;
  
  try {
    await new Promise((resolve, reject) => {
      chrome.runtime.sendMessage({ type: "CHECK_UPDATE" }, (response) => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
        } else {
          resolve(response);
        }
      });
    });
    
    button.textContent = "✅ 检查完成";
    setTimeout(() => {
      button.textContent = originalText;
      button.disabled = false;
    }, 2000);
  } catch (error) {
    console.error("检查更新失败:", error);
    button.textContent = "❌ 检查失败";
    setTimeout(() => {
      button.textContent = originalText;
      button.disabled = false;
    }, 2000);
  }
});

// ======== 报告错误 ========
$("reportError").addEventListener("click", () => {
  // 创建错误报告内容
  const manifest = chrome.runtime.getManifest();
  const userAgent = navigator.userAgent;
  const timestamp = new Date().toISOString();
  
  const errorReport = {
    extension: {
      name: manifest.name,
      version: manifest.version
    },
    browser: {
      userAgent: userAgent,
      language: navigator.language
    },
    timestamp: timestamp,
    description: "请描述您遇到的问题...",
    steps: "请描述重现问题的步骤...",
    expected: "您期望的结果是什么？",
    actual: "实际发生了什么？"
  };
  
  // 打开错误报告页面
  const reportUrl = `https://open.lingowl.com/lingowl-chat/report-error?data=${encodeURIComponent(JSON.stringify(errorReport))}`;
  chrome.tabs.create({ url: reportUrl });
  
  // 显示成功提示
  const button = $("reportError");
  const originalText = button.textContent;
  button.textContent = "✅ 已打开";
  button.style.backgroundColor = "#28a745";
  
  setTimeout(() => {
    button.textContent = originalText;
    button.style.backgroundColor = "#dc3545";
  }, 2000);
});

// ======== 获取测试 API Key ========
$("getTestKey").addEventListener("click", async (e) => {
  e.preventDefault(); // 阻止链接默认行为
  const link = $("getTestKey");
  const originalText = link.textContent;
  
  // 显示加载状态
  link.textContent = "(获取中...)";
  link.style.color = "#6c757d";
  link.style.pointerEvents = "none";
  
  try {
    const response = await fetch("https://open.lingowl.com/lingowl-chat/testapi", {
      method: "GET",
      headers: {
        "Content-Type": "application/json"
      }
    });
    
    if (response.ok) {
      const data = await response.json();
      if (data.apiKey) {
        // 将获取到的测试 Key 填入输入框
        $("apiKey").value = data.apiKey;
        $("apiKey").type = "text"; // 临时显示以便用户确认
        link.textContent = "(✅ 已获取)";
        link.style.color = "#28a745";
        
        // 2秒后恢复为密码框
        setTimeout(() => {
          $("apiKey").type = "password";
        }, 2000);
        
        // 3秒后恢复链接状态
        setTimeout(() => {
          link.textContent = originalText;
          link.style.color = "#28a745";
          link.style.pointerEvents = "auto";
        }, 3000);
      } else {
        throw new Error("未获取到 API Key");
      }
    } else {
      throw new Error(`请求失败: ${response.status}`);
    }
  } catch (error) {
    console.error("获取测试 API Key 失败:", error);
    link.textContent = "(❌ 获取失败)";
    link.style.color = "#dc3545";
    
    // 3秒后恢复链接状态
    setTimeout(() => {
      link.textContent = originalText;
      link.style.color = "#28a745";
      link.style.pointerEvents = "auto";
    }, 3000);
  }
});